#' LSMNP Posterior Mean
#'
#' This function computes the posterior mean of the parameter estimates 
#'
#' Returns the posterior means for LSMNPGibbs object as a list.  Make sure to first run 'postProcessZ()' and attach to LSMNP obj 'ZProcessed' and 'BzProcessed'
#' 
#' @param obj An object returned from LSMNPGibbs()
#' @param burnin Integer. The number of initial MCMC iterations to throw out before computing the posterior mean.
#' @param type Either 'Z' or 'Dist'.  The former will give the posterior mean of the latent positions, post-processed using the Procrustes transformation.  The latter will give the posterior mean of the distance matrix of the Z's.
#' @param thin The number by which to thin the MCMC chain.
#' @return A list.
#' \describe{
#' \item{alphak}
#' \item{sender}
#' \item{receiver}
#' \item{Sigma}
#' \item{alpha}
#' \item{sigma2}
#' \item{beta0k}
#' \item{beta1}
#' \item{Z}{(If type equals 'Z')}
#' \item{DMat}{(If type equals 'Dist')}
#' \item{sigam2Z}
#' \item{theta}
#' \item{Bs}
#' \item{Br}
#' \item{Bz}
#' }
getPostMean = function(obj, burnin=0, type=c("Z","Dist"),thin=1){
  
  nsims = length(obj$alpha)
  thSeq = seq(burnin+1,nsims,by=thin)
  # p = dim(obj$Z)[2]
  p = ifelse(length(dim(obj$Z))>2, dim(obj$Z)[2],1)
  
  if(type =="Z" & is.null(obj$ZProcessed) & (p != 1)){
    stop("Run 'postProcessZ()' and attach to LSMNP obj 'ZProcessed' and 'BzProcessed' ")
  }
  
  alphak = rowMeans(obj$alphak[,thSeq])
  sender = rowMeans(obj$sender[,thSeq])
  receiver = rowMeans(obj$receiver[,thSeq])
  Sigma = matrix(0,2,2)
  Sigma[1,1] = mean(obj$Sigma[1,1,thSeq])
  Sigma[2,1] =   Sigma[1,2] = mean(obj$Sigma[1,2,thSeq])
  Sigma[2,2] = mean(obj$Sigma[2,2,thSeq])
  alpha = mean(obj$alpha[thSeq])
  sigma2 = mean(obj$sigma2[thSeq])
  beta0k = rowMeans(obj$beta0k[,thSeq])
  beta1 = mean(obj$beta1[thSeq])
  sigma2Z = mean(obj$sigma2Z[thSeq])
  theta = mean(obj$theta[thSeq])
  Bs = rowMeans(obj$Bs[,thSeq])
  Br = rowMeans(obj$Br[,thSeq])
  if(p==1){
    Bz = rowMeans(obj$Bz[,thSeq])
  }else{
    Bz = matrix(0,nrow(obj$BzProcessed),ncol(obj$BzProcessed))
    for(ell in 1:ncol(obj$Bz)) Bz[,ell] = rowMeans(obj$BzProcessed[,ell,thSeq])
  }
  
  if(type=="Z"){
    Z = matrix(0,nrow(obj$Z),p)
    if(p == 1){
      Z[] = rowMeans(obj$Z[,1,thSeq])
    }else{
      for(ell in 1:p) Z[,ell] = rowMeans(obj$ZProcessed[,ell,thSeq])
    }
    DMat = NULL
  }else{
    DMat = matrix(0,nrow(obj$Z),nrow(obj$Z))
    if(p == 1){
      for(it in thSeq){
        DMat = DMat + Dist(matrix(obj$Z[,it],ncol=1))/length(thSeq)
      }
    }else{
      for(it in thSeq){
        DMat = DMat + Dist(obj$Z[,,it])/length(thSeq)
      }
    }
    Z = cmdscale(DMat,k = p)
  }
  
  return(list(alphak=alphak,sender=sender,receiver=receiver,Sigma=Sigma,
              alpha=alpha,sigma2=sigma2,beta0k=beta0k,beta1=beta1,
              Z=Z,DMat=DMat,sigma2Z=sigma2Z,theta=theta,Bs=Bs,Br=Br,Bz=Bz))
}
