#' LSMNP Gibbs Sampling
#'
#' The Latent Space Model for Network Perceptions (LSMNP) is a probit model that is fully described in Sewell 2019+.
#'
#' Perform Gibbs sampling for the LSM for network perception data
#' 
#' @param A A list of nxn adjacency matrices
#' @param X A nxp matrix of covariates
#' @param k.index A Kx1 integer vector giving the IDs of the respondents.  Should be the same length as A.
#' @param p The dimension of the latent space
#' @param nsims The number of simulations
#' @param quickStart A named list giving the starting positions.  Should include 
#' \describe{
#' \item{alpha}{ A scalar}
#' \item{alphak}{ A Kx1 numeric vector}
#' \item{sender}{ A nx1 numeric vector}
#' \item{receiver}{ A nx1 numeric vector}
#' \item{betak0}{ A Kx1 numeric vector}
#' \item{Z}{ A nxp matrix}
#' \item{beta1}{ A scalar}
#' \item{sigma2}{ A scalar}
#' \item{Sigma}{ A 2x2 spd matrix}
#' \item{theta}{ A scalar}
#' \item{sigma2Z}{ A scalar}
#' \item{Bs}{ A qx1 numeric vector}
#' \item{Br}{ A qx1 numeric vector}
#' \item{Bz}{ A qxp matrix}
#' }
#' @param hyperparms A named list giving the hyperparameters.  Should include 
#' \describe{
#' \item{muA}{ A scalar (default = 0)}
#' \item{nuA}{ A scalar (default = 100)}
#' \item{gammaSigma2}{ A scalar (default = 0.001)}
#' \item{etaSigma2}{ A scalar (default = 0.001)}
#' \item{gammaSig}{ A scalar (default = 2)}
#' \item{GammaSig}{ A 2x2 matrix (default = 0.001*diag(2))}
#' \item{muTheta}{ A scalar (default = 0)}
#' \item{nuTheta}{ A scalar (default = 3)}
#' \item{muB}{ A scalar (default = 0)}
#' \item{nuB}{ A scalar (default = 3)}
#' \item{gammaZ}{ A scalar (default = 0.001)}
#' \item{etaZ}{ A scalar (default = 0.001)}
#' \item{s2Bsr}{ A scalar (default = 100)}
#' \item{s2Bz}{ A scalar (default = 100)}
#' }
#' @param tuneZ Scalar. Initial tuning parameter for Z (will be updated for better acceptance rate through burn-in).
#' @param tuneB Scalar. Initial tuning parameter for beta1 (will be updated for better acceptance rate through burn-in).
#' @param tuneTheta Scalar. Initial tuning parameter for theta (will be updated for better acceptance rate through burn-in).
#' @param burnin Integer. Number of Gibbs iterations to autotune.  Note that these iterations will still be returned.
#' @param postProcess Logical. Should procrustes transformation be used?
#' @param autoTune Integer.  Number of iterations completed before checking acceptance rate.  To avoid auto-tuning, set to some integer larger than burnin.
#' @param Thin Integer.  Only every Thin-th draw will be kept.
#' @return A list.
#' \describe{
#' \item{alphak}{MCMC draws}
#' \item{sender}{MCMC draws}
#' \item{receiver}{MCMC draws}
#' \item{Sigma}{MCMC draws}
#' \item{alpha}{MCMC draws}
#' \item{sigma2}{MCMC draws}
#' \item{beta0k}{MCMC draws}
#' \item{beta1}{MCMC draws}
#' \item{Z}{MCMC draws}
#' \item{sigam2Z}{MCMC draws}
#' \item{theta}{MCMC draws}
#' \item{Bs}{MCMC draws}
#' \item{Br}{MCMC draws}
#' \item{Bz}{MCMC draws}
#' \item{logPostTrace}{Trace of the log posterior}
#' \item{tuners}{Tuning parameter values used after the burn-in period.}
#' \item{accRate}{Acceptance rates for after burn-in period.}
#' \item{hyperparms}{Hyperparameters used.}
#' }
LSMNPGibbs = function(A,X,k.index=1:length(A),p=3,nsims=1000,quickStart=NULL,
                      hyperparms=NULL,tuneZ=0.01,tuneB=1,
                      tuneTheta=0.1,burnin=100,
                      postProcess=TRUE,autoTune=burnin,Thin = 100){
  
  ###Create objects
  {
    n1 = length(A)
    n = nrow(A[[1]])
    qq = ncol(X)
    
    alpha = numeric(nsims+burnin)
    alphak = matrix(0,n1,nsims+burnin)
    sender = receiver = matrix(0,n,nsims+burnin)
    Z = array(0,c(n,p,nsims+burnin))
    beta0k = matrix(0,n1,nsims+burnin)
    beta1 = numeric(nsims+burnin)
    Bs = Br  = matrix(0,qq,nsims+burnin)
    Bz = array(0,c(qq,p,nsims+burnin))
    sigma2 = numeric(nsims+burnin)
    Sigma = array(0,c(2,2,nsims+burnin))
    theta = numeric(nsims+burnin)
    sigma2Z = numeric(nsims+burnin)
    
    logPostTrace = numeric(nsims+burnin)
  }# End: Create objects
  
  ###Create helper objects
  {
    XtX = crossprod(X)
    XtXInv = chol2inv(chol(XtX))
    Aind = list()
    for(k in 1:n1){
      Aind[[k]] = list()
      Aind[[k]][[1]] = which(A[[k]]==1,arr.ind=T)
      Aind[[k]][[2]] = which(A[[k]]==0,arr.ind=T)
      Aind[[k]][[2]] = Aind[[k]][[2]][-which(Aind[[k]][[2]][,1]==Aind[[k]][[2]][,2]),]
    }
    
  }# End: Create helper objects
  
  ###Initialize 
  cat("\n Beginning initialization. \n")
  {
    if(is.null(quickStart)){
      initDat = data.frame(Aij = rep(0,n1*n*(n-1)),
                           ak = factor(rep(1:n1,each=n*(n-1))),
                           s = factor(rep(vecNoDiag(matrix(1:n,n,n)),n1)),
                           r = factor(rep(rep(1:n,each=n-1),n1)))
      for(k in 1:n1){
        initDat$Aij[(k-1)*n*(n-1) + 1:(n*(n-1))] = 
          vecNoDiag(A[[k]])
      }
      options(contrasts=c("contr.sum","contr.poly"))
      tempMod = glm(Aij~ak+s+r,family=binomial("probit"),data=initDat,
                    control=glm.control(maxit=20))
      options(contrasts=c("contr.treatment","contr.poly"))
      
      alpha[1] = tempMod$coef[1]
      alphak[,1] = alpha[1] + c(tempMod$coef[2:n1],-sum(tempMod$coef[2:n1]))
      sender[,1] = c(tempMod$coef[n1 + 1:(n-1)],-sum(tempMod$coef[n1 + 1:(n-1)]))
      receiver[,1] = c(tempMod$coef[n1 + n-1 + 1:(n-1)],-sum(tempMod$coef[n1 + n-1 + 1:(n-1)]))
      beta0k[,1] = 1
      beta1[1] = 0.1
      
      temp = lapply(1:n1,function(i)alphak[i,1] + tcrossprod(sender[,1],receiver[,1]) - A[[i]])
      dMat = Reduce("+",temp)/n1
      dMat = dMat - min(dMat) + 1e-2
      temp = cmdscale(dMat,p)
      Z[,1:ncol(temp),1] = temp
      alpha[1] = alpha[1] + mean(dMat[upper.tri(dMat)])
      
      Bs[,1] = coef(lm(sender[,1]~X-1))
      Br[,1] = coef(lm(receiver[,1]~X-1))
      Bz[,,1] = XtXInv%*%crossprod(X,Z[,,1])
      sigma2Z[1] = var(c(Z[,,1]))
      theta[1] = 0.001
      Sigma[,,1] = cov(cbind(sender[,1],receiver[,1]))
      sigma2[1] = var(alphak[,1])
    }else{
      alpha[1] = quickStart$alpha
      alphak[,1] = quickStart$alphak
      sender[,1] = quickStart$sender
      receiver[,1] = quickStart$receiver
      beta0k[,1] = quickStart$beta0k
      Z[,,1] = quickStart$Z
      beta1[1] = quickStart$beta1
      sigma2[1] = quickStart$sigma2
      Sigma[,,1] = quickStart$Sigma
      theta[1] = quickStart$theta
      sigma2Z[1] = quickStart$sigma2Z
      Bs[,1] = quickStart$Bs
      Br[,1] = quickStart$Br
      Bz[,,1] = quickStart$Bz
    }
    
  }# End: Initialize 
  
  
  ###Hyperparameters (auto)selection
  {
    if(is.null(hyperparms$muA)){
      muA = 0
    }else{
      muA = hyperparms$muA
    }
    if(is.null(hyperparms$nuA)){
      nuA = 100
    }else{
      nuA = hyperparms$nuA
    }
    if(is.null(hyperparms$gammaSigma2)){
      gammaSigma2 = 0.001
    }else{
      gammaSigma2 = hyperparms$gammaSigma2
    }
    if(is.null(hyperparms$etaSigma2)){
      etaSigma2 = 0.001
    }else{
      etaSigma2 = hyperparms$etaSigma2
    }
    
    if(is.null(hyperparms$gammaSig)){
      gammaSig = 2
    }else{
      gammaSig = hyperparms$gammaSig
    }
    if(is.null(hyperparms$GammaSig)){
      GammaSig = 0.001*diag(2)
    }else{
      GammaSig = hyperparms$GammaSig
    }
    
    if(is.null(hyperparms$nuTheta)){
      nuTheta = 3
    }else{
      nuTheta = hyperparms$nuTheta
    }
    if(is.null(hyperparms$muTheta)){
      muTheta = 0
    }else{
      muTheta = hyperparms$muTheta
    }
    
    if(is.null(hyperparms$muB)){
      muB = 0
    }else{
      muB = hyperparms$muB
    }
    if(is.null(hyperparms$nuB)){
      nuB = 3
    }else{
      nuB = hyperparms$nuB
    }
    
    if(is.null(hyperparms$gammaZ)){
      gammaZ = 0.001
    }else{
      gammaZ = hyperparms$gammaZ
    }
    if(is.null(hyperparms$etaZ)){
      etaZ = 0.001
    }else{
      etaZ = hyperparms$etaZ
    }
    
    if(is.null(hyperparms$s2Bsr)){
      s2Bsr = 100
    }else{
      s2Bsr = hyperparms$s2Bsr
    }
    if(is.null(hyperparms$s2Bz)){
      s2Bz = 100
    }else{
      s2Bz = hyperparms$s2Bz
    }
  }# End: Hyperparameters
  
  ###More helper objects
  {
    if(p == 1){
      DMat = Dist(matrix(Z[,,1],nc=1))
    }else{
      DMat = Dist(Z[,,1])
    }
    sr = matrix(sender[,1],n,n) + matrix(receiver[,1],n,n,byrow=TRUE)
    tau = getTau(beta0k[,1],beta1[1],DMat,k.index-1)
    
    Sigsr = matrix(0,2*n,2*n)
    musr= numeric(2*n)
    
    Astar = list()
    for(k in 1:n1){
      Astar[[k]] = alphak[k] + sr - DMat
      Astar[[k]][Aind[[k]][[1]]] =
        rtruncnorm(nrow(Aind[[k]][[1]]),a=0,
                   mean=Astar[[k]][Aind[[k]][[1]]],
                   sd=sqrt(1/tau[,,k][Aind[[k]][[1]]]))
      Astar[[k]][Aind[[k]][[2]]] =
        rtruncnorm(nrow(Aind[[k]][[2]]),b=0,
                   mean=Astar[[k]][Aind[[k]][[2]]],
                   sd=sqrt(1/tau[,,k][Aind[[k]][[2]]]))
      diag(Astar[[k]]) = 0
      diag(tau[,,k]) = 0
    }
    
    gammaSig.it = gammaSig + n
    gammaSigma2.it = gammaSigma2 + n1
    nuA.it = nuA/(1+nuA*n1)
    nuB.it = nuB/(1+nuB*n1)
    gammaZ.it = gammaZ + n*p
    
  }# End: More helper objects  
  
  ### Prepare for burn-in
  {
    logPostTrace[1] = LogPost(A,X,Astar,k.index,
                              alphak[,1],sender[,1],receiver[,1],
                              Z[,,1],beta0k[,1],beta1[1],tau,
                              Bs[,1],Br[,1],Bz[,,1],sigma2Z[1],
                              alpha[1],sigma2[1],Sigma[,,1],
                              theta[1],muA, nuA,gammaSigma2, 
                              etaSigma2, gammaSig, GammaSig, 
                              muB, nuB, gammaZ,etaZ, s2Bsr, s2Bz)
    
    tuneUB.Z=tuneLB.Z=tuneZ
    accRateVec.Z=0
    ARLB.Z=ARUB.Z=0
    numTries.Z = 0
    const=2
    tuningTempVec.Z=numeric(burnin)
    tuningTempVec.Z[1] = tuneZ
    trend.Z=0
    
    tuneUB.B=tuneLB.B=tuneB
    accRateVec.B=ARLB.B=ARUB.B=0
    numTries.B = 0
    tuningTempVec.B=numeric(burnin)
    tuningTempVec.B[1] = tuneB
    trend.B=0
    
    tuneUB.Th=tuneLB.Th=tuneTheta
    accRateVec.Th=ARLB.Th=ARUB.Th=0
    numTries.Th = 0
    tuningTempVec.Th=numeric(burnin)
    tuningTempVec.Th[1] = tuneTheta
    trend.Th=0
    
    Sig.sr.fill1 = cbind(1:n,1:n)
    Sig.sr.fill2 = cbind(1:n,n+1:n)
    Sig.sr.fill3 = cbind(n+1:n,1:n)
    Sig.sr.fill4 = cbind(n+1:n,n+1:n)
    
    SigInv = 1/(Sigma[1,1,1]*Sigma[2,2,1]-Sigma[1,2,1]*Sigma[2,1,1])*
      matrix(Sigma[,,1][c(4,2,3,1)]*c(1,-1,-1,1),2,2)
    Sigsr[] = 0
    Sigsr[Sig.sr.fill1] = SigInv[1,1]
    Sigsr[Sig.sr.fill2] = SigInv[1,2]
    Sigsr[Sig.sr.fill3] = SigInv[2,1]
    Sigsr[Sig.sr.fill4] = SigInv[2,2]
    
    
    XBz = X%*%Bz[,,1]
    
  }# End: Prepare for burn-in
  
  ###
  ###burn-in
  ###
  cat("\n MCMC Sampling: Burn in period and auto-tuning. \n")
  {
    pb = txtProgressBar(min=2,max=burnin,style=3)
    for(it in 2:burnin){
      alphak[,it] = alphak[,it-1]
      sender[,it] = sender[,it-1]
      receiver[,it] = receiver[,it-1]
      Bs[,it] = Bs[,it-1]
      Br[,it] = Br[,it-1]
      Sigma[,,it] = Sigma[,,it-1]
      alpha[it] = alpha[it-1]
      sigma2[it] = sigma2[it-1]
      Bz[,,it] = Bz[,,it-1]
      sigma2Z[it] = sigma2Z[it-1]
      Z[,,it] = Z[,,it-1]
      beta0k[,it] = beta0k[,it-1]
      beta1[it] = beta1[it-1]
      theta[it] = theta[it-1]
      
      for(itTh in 1:Thin){
        ###Random scan:
        updateInd = sample.int(12,1)
        
        ###alpha_k
        if(updateInd == 1){
          s2ak = 1/( 1/sigma2[it] + cAddOffDiags(tau))
          muak = numeric(n1)
          for(k in 1:n1){
            muak[k] = s2ak[k]*(alpha[it]/sigma2[it]+
                                 cSumMatMultNoDiag(tau[,,k],Astar[[k]],DMat,
                                                   sender[,it],receiver[,it]))
          }
          alphak[,it] = rnorm(n1,muak,sqrt(s2ak))
        }
        
        ###Sender/receiver effects
        if(updateInd == 2){
          SigInv = 1/(Sigma[1,1,it]*Sigma[2,2,it]-Sigma[1,2,it]*Sigma[2,1,it])*
            matrix(Sigma[,,it][c(4,2,3,1)]*c(1,-1,-1,1),2,2)
          Sigsr[] = 0
          Sigsr[Sig.sr.fill1] = SigInv[1,1]
          Sigsr[Sig.sr.fill2] = SigInv[1,2]
          Sigsr[Sig.sr.fill3] = SigInv[2,1]
          Sigsr[Sig.sr.fill4] = SigInv[2,2]
          for(k in 1:n1){
            Sigsr[Sig.sr.fill1] = Sigsr[Sig.sr.fill1] + rowSums(tau[,,k])
            Sigsr[Sig.sr.fill4] = Sigsr[Sig.sr.fill4] + colSums(tau[,,k])
            Sigsr[1:n,n+1:n] = Sigsr[1:n,n+1:n] + tau[,,k]
            Sigsr[n+1:n,1:n] = Sigsr[n+1:n,1:n] + tau[,,k]
          }
          Sigsr = chol2inv(chol(Sigsr))
          musr = getMusr(Astar,DMat,alphak[,it],tau,SigInv,X,c(Bs[,it],Br[,it]),Sigsr)
          temp = mvtnorm::rmvnorm(1,mean=musr,sigma=Sigsr)
          sender[,it] = temp[1:n]
          receiver[,it] = temp[n+1:n]
          
          tempMeans = c(mean(sender[,it]),mean(receiver[,it]))
          sender[,it] = sender[,it] - tempMeans[1]
          receiver[,it] = receiver[,it] - tempMeans[2]
          alphak[,it] = alphak[,it] + sum(tempMeans)
          sr = matrix(sender[,it],n,n) + matrix(receiver[,it],n,n,byrow=TRUE)
        }
        
        ###Bs and Br
        if(updateInd == 3){
          temp1 = getSigBsr(SigInv,XtX,s2Bsr,X,c(sender[,it],receiver[,it]))
          temp2 = drop(rmvnorm(1,mean=temp1$muBsr,sigma=temp1$SigBsr))
          Bs[,it] = temp2[1:qq]
          Br[,it] = temp2[qq+1:qq]
        }
        
        ###Sigma
        if(updateInd == 4){
          sminXBs = sender[,it]-X%*%Bs[,it]
          rminXBr = receiver[,it]-X%*%Br[,it]
          GammaSig.it = GammaSig + crossprod(cbind(sminXBs,rminXBr))
          Sigma[,,it] = riwish(gammaSig.it,GammaSig.it)
        }
        
        ###alpha and sigma2
        if(updateInd == 5){
          muA.it = (muA + nuA*sum(alphak[,it]))/(1+nuA*n1)
          etaSigma2.it = etaSigma2 + sum(alphak[,it]^2) + muA^2/nuA - muA.it^2/nuA.it
          sigma2[it] = rinvgamma(1,shape=gammaSigma2.it/2,scale=etaSigma2.it/2)
          alpha[it] = rnorm(1,muA.it,sqrt(nuA.it*sigma2[it]))
        }
        
        ###Bz
        if(updateInd == 6){
          if(p == 1){
            Bz[,,it] = cUzMz(sigma2Z[it],XtX,s2Bz,X,matrix(Z[,,it],nc=1))$Bz
          }else{
            Bz[,,it] = cUzMz(sigma2Z[it],XtX,s2Bz,X,Z[,,it])$Bz
          }
        }
        
        ###sigma2Z
        if(updateInd == 7){
          XBz = X%*%Bz[,,it]
          etaZ.it = etaZ + norm(Z[,,it]-XBz,type="F")^2
          sigma2Z[it] = rinvgamma(1,shape=gammaZ.it/2,etaZ.it/2)
        }
        
        ###Z
        if(updateInd == 8){
          numTries.Z = numTries.Z + 1
          ZNew = Z[,,it] + matrix(rnorm(n*p,sd=tuneZ),n,p)
          if(p == 1){
            DMatNew = Dist(matrix(ZNew,nc=1))
          }else{
            DMatNew = Dist(ZNew)
          }
          tauNew = getTau(beta0k[,it],beta1[it],DMatNew,k.index - 1)
          AccProb =  exp(
            LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                   receiver[,it],DMatNew,beta0k[,it],beta1[it],tauNew) -
              LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                     receiver[,it],DMat,beta0k[,it],beta1[it],tau) -
              0.5*(norm(ZNew-XBz,type="F")^2-norm(Z[,,it]-XBz,type="F")^2)/sigma2Z[it])
          
          if(runif(1) < AccProb){
            accRateVec.Z = accRateVec.Z + 1
            Z[,,it] = ZNew
            DMat = DMatNew
            tau = tauNew
          }
        }
        
        ###betak0
        if(updateInd == 9){
          theta1k = theta[it] + n*(n-1)
          theta2k = cGetTheta2k(tau,Astar,DMat,sender[,it],
                                receiver[,it],beta0k[,it],
                                alphak[,it],theta[it])
          beta0kNew = rgamma(n1,shape=theta1k/2,rate=theta2k)
          # burn-in only:
          beta0kNew = beta0kNew/mean(beta0kNew)
          tau = cUpdateTauWithBeta0(tau,beta0kNew,beta0k[,it])
          beta0k[,it] = beta0kNew
        }
        
        ###beta1
        if(updateInd == 10){
          numTries.B = numTries.B + 1
          beta1New = beta1[it]*exp(rnorm(1,sd=tuneB))
          tauNew = getTau(beta0k[,it],beta1New,DMat,k.index-1)
          AccProb = exp(
            LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                   receiver[,it],DMat,beta0k[,it],beta1New,tauNew) -
              LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                     receiver[,it],DMat,beta0k[,it],beta1[it],tau) -
              0.5*sum((log(beta1New)-muB)^2-(log(beta1[it])-muB)^2)/nuB )
          if(runif(1) < AccProb){
            accRateVec.B = accRateVec.B + 1
            beta1[it] = beta1New
            tau = tauNew
          }
        }
        
        ###theta
        if(updateInd == 11){
          numTries.Th = numTries.Th + 1
          thetaNew = theta[it]*exp(rnorm(1,sd=tuneTheta))
          AccProb = exp(
            sum( dgamma(beta0k[,it],shape=thetaNew/2,rate=thetaNew/2,log=T) -
                   dgamma(beta0k[,it],shape=theta[it]/2,rate=theta[it]/2,log=T) ) +
              sum( dnorm(log(thetaNew),mean=muTheta,sd=sqrt(nuTheta),log=T) -
                     dnorm(log(theta[it]),mean=muTheta,sd=sqrt(nuTheta),log=T)  ) )
          if(runif(1) < AccProb){
            theta[it] = thetaNew
            accRateVec.Th = accRateVec.Th + 1
          }
          # theta[it] = 0.001
        }
        
        ###Astar
        if(updateInd == 12){
          for(k in 1:n1){
            Astar[[k]] = alphak[k,it] + sr - DMat
            Astar[[k]][Aind[[k]][[1]]] = 
              rtruncnorm(nrow(Aind[[k]][[1]]),a=0,
                         mean=Astar[[k]][Aind[[k]][[1]]],
                         sd=sqrt(1/tau[,,k][Aind[[k]][[1]]]))
            Astar[[k]][Aind[[k]][[2]]] = 
              rtruncnorm(nrow(Aind[[k]][[2]]),b=0,
                         mean=Astar[[k]][Aind[[k]][[2]]],
                         sd=sqrt(1/tau[,,k][Aind[[k]][[2]]]))
            
          }
        }
      }
      
      logPostTrace[it] = 
        LogPost(A,X,Astar,k.index,
                alphak[,it],sender[,it],receiver[,it],
                Z[,,it],beta0k[,it],beta1[it],tau,
                Bs[,it],Br[,it],Bz[,,it],sigma2Z[it],
                alpha[it],sigma2[it],Sigma[,,it],
                theta[it],muA, nuA,gammaSigma2, 
                etaSigma2, gammaSig, GammaSig, 
                muB, nuB, gammaZ,etaZ, s2Bsr, s2Bz)
      
      ###Adjust tuning parameters
      if(it%%autoTune == 0){
        accRateVec.B = accRateVec.B/numTries.B
        accRateVec.Th = accRateVec.Th/numTries.Th
        accRateVec.Z = accRateVec.Z/numTries.Z
        
        if(accRateVec.Z>0.1 & accRateVec.Z<0.4){
          ARUB.Z=ARLB.Z=accRateVec.Z
          tuneUB.Z=tuneLB.Z=tuneZ
          trend.Z=0
        }else{
          if(accRateVec.Z<0.1){
            tuneLB.Z=tuneZ
            ARLB.Z = accRateVec.Z
            trend.Z=trend.Z-1
            if(ARUB.Z < 0.4 | trend.Z <= -3){
              tuneZ=tuneZ/const
            }else{
              tuneZ=0.5*(tuneLB.Z+tuneUB.Z)
            }
          }else{ #if accRateVec[itBurn]>0.4
            tuneUB.Z=tuneZ
            ARUB.Z = accRateVec.Z
            trend.Z=trend.Z+1
            if(ARLB.Z>0.1 | trend.Z>=3){
              tuneZ=tuneZ*const
            }else{
              tuneZ=0.5*(tuneLB.Z+tuneUB.Z)
            }
          }
        }
        tuningTempVec.Z[it] = tuneZ
        accRateVec.Z = numTries.Z = 0
        
        if(accRateVec.B>0.1 & accRateVec.B<0.4){
          ARUB.B=ARLB.B=accRateVec.B
          tuneUB.B=tuneLB.B=tuneB
          trend.B=0
        }else{
          if(accRateVec.B<0.1){
            tuneLB.B=tuneB
            ARLB.B = accRateVec.B
            trend.B=trend.B-1
            if(ARUB.B < 0.4 | trend.B <= -3){
              tuneB=tuneB/const
            }else{
              tuneB=0.5*(tuneLB.B+tuneUB.B)
            }
          }else{ #if accRateVec[itBurn]>0.4
            tuneUB.B=tuneB
            ARUB.B = accRateVec.B
            trend.B=trend.B+1
            if(ARLB.B>0.1 | trend.B>=3){
              tuneB=tuneB*const
            }else{
              tuneB=0.5*(tuneLB.B+tuneUB.B)
            }
          }
        }
        tuningTempVec.B[it] = tuneB
        accRateVec.B = numTries.B = 0
        
        if(accRateVec.Th>0.1 & accRateVec.Th<0.4){
          ARUB.Th=ARLB.Th=accRateVec.Th
          tuneUB.Th=tuneLB.Th=tuneTheta
          trend.Th=0
        }else{
          if(accRateVec.Th<0.1){
            tuneLB.Th=tuneTheta
            ARLB.Th = accRateVec.Th
            trend.Th=trend.Th-1
            if(ARUB.Th < 0.4 | trend.Th <= -3){
              tuneTheta=tuneTheta/const
            }else{
              tuneTheta=0.5*(tuneLB.Th+tuneUB.Th)
            }
          }else{ #if accRateVec[itBurn]>0.4
            tuneUB.Th=tuneTheta
            ARUB.Th = accRateVec.Th
            trend.Th=trend.Th+1
            if(ARLB.Th>0.1 | trend.Th>=3){
              tuneTheta=tuneTheta*const
            }else{
              tuneTheta=0.5*(tuneLB.Th+tuneUB.Th)
            }
          }
        }
        tuningTempVec.Th[it] = tuneTheta
        accRateVec.Th = numTries.Th = 0
        
      }
      
      setTxtProgressBar(pb,it)
    }
  }# End: Burn-in
  
  ###
  ###Post burn-in
  ###
  cat("\n MCMC Sampling: Post-burn in samples. \n")
  {
    accRateVec.Z = numTries.Z = 0
    accRateVec.B = numTries.B = 0
    accRateVec.Th = numTries.Th = 0
    
    if(postProcess) Zreference = scale(Z[,,burnin],scale=F)
    
    pb = txtProgressBar(min=2,max=nsims,style=3)
    for(it in burnin+1:nsims){
      
      alphak[,it] = alphak[,it-1]
      sender[,it] = sender[,it-1]
      receiver[,it] = receiver[,it-1]
      Bs[,it] = Bs[,it-1]
      Br[,it] = Br[,it-1]
      Sigma[,,it] = Sigma[,,it-1]
      alpha[it] = alpha[it-1]
      sigma2[it] = sigma2[it-1]
      Bz[,,it] = Bz[,,it-1]
      sigma2Z[it] = sigma2Z[it-1]
      Z[,,it] = Z[,,it-1]
      beta0k[,it] = beta0k[,it-1]
      beta1[it] = beta1[it-1]
      theta[it] = theta[it-1]
      
      for(itTh in 1:Thin){
        ###Random scan:
        updateInd = sample.int(12,1)
        
        ###alpha_k
        if(updateInd == 1){
          s2ak = 1/( 1/sigma2[it] + cAddOffDiags(tau))
          muak = numeric(n1)
          for(k in 1:n1){
            muak[k] = s2ak[k]*(alpha[it]/sigma2[it]+
                                 cSumMatMultNoDiag(tau[,,k],Astar[[k]],DMat,
                                                   sender[,it],receiver[,it]))
          }
          alphak[,it] = rnorm(n1,muak,sqrt(s2ak))
        }
        
        ###Sender/receiver effects
        if(updateInd == 2){
          SigInv = 1/(Sigma[1,1,it]*Sigma[2,2,it]-Sigma[1,2,it]*Sigma[2,1,it])*
            matrix(Sigma[,,it][c(4,2,3,1)]*c(1,-1,-1,1),2,2)
          Sigsr[] = 0
          Sigsr[Sig.sr.fill1] = SigInv[1,1]
          Sigsr[Sig.sr.fill2] = SigInv[1,2]
          Sigsr[Sig.sr.fill3] = SigInv[2,1]
          Sigsr[Sig.sr.fill4] = SigInv[2,2]
          for(k in 1:n1){
            Sigsr[Sig.sr.fill1] = Sigsr[Sig.sr.fill1] + rowSums(tau[,,k])
            Sigsr[Sig.sr.fill4] = Sigsr[Sig.sr.fill4] + colSums(tau[,,k])
            Sigsr[1:n,n+1:n] = Sigsr[1:n,n+1:n] + tau[,,k]
            Sigsr[n+1:n,1:n] = Sigsr[n+1:n,1:n] + tau[,,k]
          }
          Sigsr = chol2inv(chol(Sigsr))
          musr = getMusr(Astar,DMat,alphak[,it],tau,SigInv,X,c(Bs[,it],Br[,it]),Sigsr)
          temp = mvtnorm::rmvnorm(1,mean=musr,sigma=Sigsr)
          sender[,it] = temp[1:n]
          receiver[,it] = temp[n+1:n]
          
          tempMeans = c(mean(sender[,it]),mean(receiver[,it]))
          sender[,it] = sender[,it] - tempMeans[1]
          receiver[,it] = receiver[,it] - tempMeans[2]
          alphak[,it] = alphak[,it] + sum(tempMeans)
          sr = matrix(sender[,it],n,n) + matrix(receiver[,it],n,n,byrow=TRUE)
        }
        
        ###Bs and Br
        if(updateInd == 3){
          temp1 = getSigBsr(SigInv,XtX,s2Bsr,X,c(sender[,it],receiver[,it]))
          temp2 = drop(rmvnorm(1,mean=temp1$muBsr,sigma=temp1$SigBsr))
          Bs[,it] = temp2[1:qq]
          Br[,it] = temp2[qq+1:qq]
        }
        
        ###Sigma
        if(updateInd == 4){
          sminXBs = sender[,it]-X%*%Bs[,it]
          rminXBr = receiver[,it]-X%*%Br[,it]
          GammaSig.it = GammaSig + crossprod(cbind(sminXBs,rminXBr))
          Sigma[,,it] = riwish(gammaSig.it,GammaSig.it)
        }
        
        ###alpha and sigma2
        if(updateInd == 5){
          muA.it = (muA + nuA*sum(alphak[,it]))/(1+nuA*n1)
          etaSigma2.it = etaSigma2 + sum(alphak[,it]^2) + muA^2/nuA - muA.it^2/nuA.it
          sigma2[it] = rinvgamma(1,shape=gammaSigma2.it/2,scale=etaSigma2.it/2)
          alpha[it] = rnorm(1,muA.it,sqrt(nuA.it*sigma2[it]))
        }
        
        ###Bz
        if(updateInd == 6){
          if(p == 1){
            Bz[,,it] = cUzMz(sigma2Z[it],XtX,s2Bz,X,matrix(Z[,,it],nc=1))$Bz
          }else{
            Bz[,,it] = cUzMz(sigma2Z[it],XtX,s2Bz,X,Z[,,it])$Bz
          }
        }
        
        ###sigma2Z
        if(updateInd == 7){
          XBz = X%*%Bz[,,it]
          etaZ.it = etaZ + norm(Z[,,it]-XBz,type="F")^2
          sigma2Z[it] = rinvgamma(1,shape=gammaZ.it/2,etaZ.it/2)
        }
        
        ###Z
        if(updateInd == 8){
          numTries.Z = numTries.Z + 1
          ZNew = Z[,,it] + matrix(rnorm(n*p,sd=tuneZ),n,p)
          if(p == 1){
            DMatNew = Dist(matrix(ZNew,nc=1))
          }else{
            DMatNew = Dist(ZNew)
          }
          tauNew = getTau(beta0k[,it],beta1[it],DMatNew,k.index - 1)
          AccProb =  exp(
            LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                   receiver[,it],DMatNew,beta0k[,it],beta1[it],tauNew) -
              LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                     receiver[,it],DMat,beta0k[,it],beta1[it],tau) -
              0.5*(norm(ZNew-XBz,type="F")^2-norm(Z[,,it]-XBz,type="F")^2)/sigma2Z[it])
          
          if(runif(1) < AccProb){
            accRateVec.Z = accRateVec.Z + 1
            Z[,,it] = ZNew
            DMat = DMatNew
            tau = tauNew
          }
        }
        
        ###betak0
        if(updateInd == 9){
          theta1k = theta[it] + n*(n-1)
          theta2k = cGetTheta2k(tau,Astar,DMat,sender[,it],
                                receiver[,it],beta0k[,it],
                                alphak[,it],theta[it])
          beta0kNew = rgamma(n1,shape=theta1k/2,rate=theta2k)
          # burn-in only:
          beta0kNew = beta0kNew/mean(beta0kNew)
          tau = cUpdateTauWithBeta0(tau,beta0kNew,beta0k[,it])
          beta0k[,it] = beta0kNew
        }
        
        ###beta1
        if(updateInd == 10){
          numTries.B = numTries.B + 1
          beta1New = beta1[it]*exp(rnorm(1,sd=tuneB))
          tauNew = getTau(beta0k[,it],beta1New,DMat,k.index-1)
          AccProb = exp(
            LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                   receiver[,it],DMat,beta0k[,it],beta1New,tauNew) -
              LogLik(A,Aind,k.index,alphak[,it],sender[,it],
                     receiver[,it],DMat,beta0k[,it],beta1[it],tau) -
              0.5*sum((log(beta1New)-muB)^2-(log(beta1[it])-muB)^2)/nuB )
          if(runif(1) < AccProb){
            accRateVec.B = accRateVec.B + 1
            beta1[it] = beta1New
            tau = tauNew
          }
        }
        
        ###theta
        if(updateInd == 11){
          numTries.Th = numTries.Th + 1
          thetaNew = theta[it]*exp(rnorm(1,sd=tuneTheta))
          AccProb = exp(
            sum( dgamma(beta0k[,it],shape=thetaNew/2,rate=thetaNew/2,log=T) -
                   dgamma(beta0k[,it],shape=theta[it]/2,rate=theta[it]/2,log=T) ) +
              sum( dnorm(log(thetaNew),mean=muTheta,sd=sqrt(nuTheta),log=T) -
                     dnorm(log(theta[it]),mean=muTheta,sd=sqrt(nuTheta),log=T)  ) )
          if(runif(1) < AccProb){
            theta[it] = thetaNew
            accRateVec.Th = accRateVec.Th + 1
          }
          # theta[it] = 0.001
        }
        
        ###Astar
        if(updateInd == 12){
          for(k in 1:n1){
            Astar[[k]] = alphak[k,it] + sr - DMat
            Astar[[k]][Aind[[k]][[1]]] = 
              rtruncnorm(nrow(Aind[[k]][[1]]),a=0,
                         mean=Astar[[k]][Aind[[k]][[1]]],
                         sd=sqrt(1/tau[,,k][Aind[[k]][[1]]]))
            Astar[[k]][Aind[[k]][[2]]] = 
              rtruncnorm(nrow(Aind[[k]][[2]]),b=0,
                         mean=Astar[[k]][Aind[[k]][[2]]],
                         sd=sqrt(1/tau[,,k][Aind[[k]][[2]]]))
            
          }
        }
      }
      
      logPostTrace[it] = 
        LogPost(A,X,Astar,k.index,
                alphak[,it],sender[,it],receiver[,it],
                Z[,,it],beta0k[,it],beta1[it],tau,
                Bs[,it],Br[,it],Bz[,,it],sigma2Z[it],
                alpha[it],sigma2[it],Sigma[,,it],
                theta[it],muA, nuA,gammaSigma2, 
                etaSigma2, gammaSig, GammaSig, 
                muB, nuB, gammaZ,etaZ, s2Bsr, s2Bz)
      
      setTxtProgressBar(pb,it-burnin)
    }
    
    accRateVec.B = accRateVec.B/numTries.B
    accRateVec.Th = accRateVec.Th/numTries.Th
    accRateVec.Z = accRateVec.Z/numTries.Z
    
  }# End: Post burn-in
  
  return(list(alphak=alphak,sender=sender,receiver=receiver,Sigma=Sigma,
              alpha=alpha,sigma2=sigma2,beta0k=beta0k,beta1=beta1,
              Z=Z,sigma2Z=sigma2Z,theta=theta,Bs=Bs,Br=Br,Bz=Bz,
              logPostTrace=logPostTrace,
              tuners = list(Z=tuneZ,beta1=tuneB,theta=tuneTheta),
              accRate = list(Z=accRateVec.Z,beta1=accRateVec.B,theta=accRateVec.Th),
              hyperparms = list(muA=muA,nuA=nuA,gammaSigma2=gammaSigma2,etaSigma2=etaSigma2,
                                gammaSig=gammaSig,GammaSig=GammaSig,muTheta=muTheta,nuTheta=nuTheta,
                                muB=muB,nuB=nuB,gammaZ=gammaZ,etaZ=etaZ,s2Bsr=s2Bsr,s2Bz=s2Bz)))
}
