#' Deviance Information Criterion for the LSMNP
#'
#' This function computes the DIC for the LSMNP model. 
#'
#' Returns the DIC and effective number of parameter. Make sure to first run 'postProcessZ()' and attach to LSMNP obj 'ZProcessed' and 'BzProcessed'
#' 
#' @param obj An object returned from LSMNPGibbs()
#' @param burnin Integer. The number of initial MCMC iterations to throw out.
#' @return A list with 'ZProcessed' and 'BzProcessed'
postProcessZ = function(obj,burnin=0){
  p = ifelse(length(dim(obj$Z))==3,dim(obj$Z)[2],1)
  if(p == 1){
    stop("Post processing procrustes transformations not needed for p = 1.")
  }
  
  nsims = length(obj$alpha)
  refIter = burnin + which.max(obj$logPostTrace[(burnin+1):nsims])
  ZRef = obj$Z[,,refIter]
  ZProcessed = obj$Z
  BzProcessed = obj$Bz
  
  
  for(it in (burnin+1):nsims){
    procr = vegan::procrustes(X=ZRef,Y=ZProcessed[,,it],scale=FALSE)$rot
    ZProcessed[,,it] = ZProcessed[,,it]%*%procr
    BzProcessed[,,it] = BzProcessed[,,it]%*%procr
  }
  return(list(ZProcessed = ZProcessed,
              BzProcessed = BzProcessed))
}
