#' LSMNP Log Posterior
#'
#' This function evaluates the log posterior for the LSMNP model.  Primarily for internal use.
#'
#' @description Log posterior for LSMNP
#' @param A A list of nxn adjacency matrices
#' @param X A nxp matrix of covariates
#' @param Astar A nxn matrix of latent normal random variables for the probit model
#' @param k.index A Kx1 integer vector giving the IDs of the respondents.  Should be the same length as A.
#' @param alphak A Kx1 numeric vector
#' @param ss A nx1 numeric vector
#' @param rr A nx1 numeric vector
#' @param Z A nxp matrix
#' @param b0k A Kx1 numeric vector
#' @param tau (optional) A nxnxK array
#' @param Bs A qx1 numeric vector
#' @param Br A qx1 numeric vector
#' @param Bz A qxp matrix
#' @param sigma2Z A scalar
#' @param Sigma A 2x2 matrix
#' @param theta A scalar
#' @param muA A scalar
#' @param nuA A scalar
#' @param gammaSigma2 A scalar
#' @param etaSigma2  A scalar
#' @param gammaSig  A scalar
#' @param GammaSig A 2x2 matrix
#' @param muB  A scalar
#' @param nuB  A scalar
#' @param gammaZ  A scalar
#' @param etaZ  A scalar
#' @param s2Bsr  A scalar
#' @param s2Bz  A scalar
#' @return A scalar providing the log posterior.
LogPost = function(A,X,Astar,k.index,alphak,ss,rr,Z,b0k,b1,tau=NULL,
                   Bs,Br,Bz,sigma2Z,alpha,sigma2,Sigma,theta,
                   muA,nuA,gammaSigma2,etaSigma2,gammaSig,
                   GammaSig,muB,nuB,gammaZ,etaZ,s2Bsr,s2Bz){
  # browser()
  ret = 0
  pp = ncol(Z)
  if(is.null(pp))pp=1
  n1 = length(Astar)
  n = nrow(A[[1]])
  if(pp == 1){
    DMat = Dist(matrix(Z,nc=1))
  }else{
    DMat = Dist(Z)
  }
  sr = matrix(ss,n,n) + matrix(rr,n,n,byrow=TRUE)
  
  if(is.null(tau)) tau = getTau(b0k,b1,DMat,k.index - 1)
  
  srMinusD = sr - DMat
  ret = 0
  for(k in 1:n1){
    ret = ret + 
      sum(dnorm(Astar[[k]][upper.tri(Astar[[k]])],
                mean=alphak[k] + srMinusD[upper.tri(Astar[[k]])],
                sd=1/sqrt(tau[,,k][upper.tri(Astar[[k]])]),log=TRUE)) +
      sum(dnorm(Astar[[k]][lower.tri(Astar[[k]])],
                mean=alphak[k] + srMinusD[lower.tri(Astar[[k]])],
                sd=1/sqrt(tau[,,k][lower.tri(Astar[[k]])]),log=TRUE))
  }
  ret = ret + sum(dnorm(Z,mean=X%*%Bz,sd=sqrt(sigma2Z),log=TRUE)) +
    sum(dnorm(alphak,alpha,sqrt(sigma2),log=TRUE)) +
    sum(dmvnorm(cbind(ss,rr)-X%*%cbind(Bs,Br),sigma=Sigma,log=TRUE)) +
    sum(dgamma(b0k,shape=theta/2,rate=theta/2,log=TRUE)) +
    log(dinvgamma(sigma2,gammaSigma2/2,etaSigma2/2)) +
    dnorm(alpha,muA,sqrt(nuA*sigma2),log=TRUE) +
    log(diwish(Sigma,gammaSig,GammaSig)) +
    dlnorm(b1,muB,sqrt(nuB),log=TRUE) +
    log(dinvgamma(sigma2Z,gammaZ/2,etaZ/2)) +
    sum(dnorm(c(Bs,Br),sd=sqrt(s2Bsr),log=TRUE)) +
    sum(dnorm(c(Bz),sd=sqrt(s2Bz),log=TRUE))
  
  return(ret)
}
