#' LSMNP Log Likelihood
#'
#' This function evaluates the log likelihood for the LSMNP model.  Primarily for internal use.
#'
#' @description Log posterior for LSMNP
#' @param A A list of nxn adjacency matrices
#' @param A.ind (optional) A list of length K, each element is itself a list of length two, the first element being the indices of A[[k]] equaling 1, the second being the indices of A[[k]] equaling 0 (excluding diagonal elements).
#' @param k.index A Kx1 integer vector giving the IDs of the respondents.  Should be the same length as A.
#' @param alphak A Kx1 numeric vector
#' @param ss A nx1 numeric vector
#' @param rr A nx1 numeric vector
#' @param Z A nxp matrix
#' @param b0k A Kx1 numeric vector
#' @param b1 A scalar
#' @param tau (optional) A nxnxK array
#' @return A scalar providing the log posterior.
LogLik = function(A,Aind = NULL,k.index,alphak,ss,rr,Z,b0k,b1,tau=NULL){
  ret = 0
  p = ncol(Z)
  n1 = length(A)
  if(is.null(Aind)){
    Aind = list()
    for(k in 1:n1){
      Aind[[k]] = list()
      Aind[[k]][[1]] = which(A[[k]]==1,arr.ind=T)
      Aind[[k]][[2]] = which(A[[k]]==0,arr.ind=T)
      Aind[[k]][[2]] = Aind[[k]][[2]][-which(Aind[[k]][[2]][,1]==Aind[[k]][[2]][,2]),]
    }
  }
  n = nrow(A[[1]])
  if(ncol(Z) < nrow(Z)){
    if(p == 1){
      DMat = Dist(matrix(Z,nc=1))
    }else{
      DMat = Dist(Z)
    }
  }else{
    DMat = Z
  }
  sr = matrix(ss,n,n) + matrix(rr,n,n,byrow=TRUE)
  
  if(is.null(tau)) tau = getTau(b0k,b1,DMat,k.index - 1)
  
  srMinusD = sr - DMat
  for(i in 1:n1){
    phi = pnorm(sqrt(tau[,,i])*(alphak[i]+srMinusD))
    ret = ret + sum(log(phi[Aind[[i]][[1]]]))
    ret = ret + sum(log(1-phi[Aind[[i]][[2]]]))
  }
  
  return(ret)
}
