#' Locally Aggregated Probabilities (LAP) for LSMNP
#'
#' This function computes the LAPs from an LSMNP model fit. 
#'
#' Returns the row dominated, column dominated, intersection, and union LAP from an LSMNP object.
#' 
#' @param k.index A Kx1 integer vector giving the IDs of the respondents.  Should be the same length as A.
#' @param obj An object returned from LSMNPGibbs()
#' @param burnin Integer. The number of initial MCMC iterations to throw out.
#' @param Thin The number by which to thin the MCMC chain.
#' @return A list with the four LAPS: row dominated, column dominated, intersection, and union LAP.
LAP = function(k.index,obj,burnin=0,Thin=1){
  nsims = length(obj$alpha)
  thSeq = seq(burnin+1,nsims,by=Thin)
  sLen = length(thSeq)
  n = nrow(obj$Z)
  n1 = length(k.index)
  AOut = AIn = matrix(0,n,n)
  
  if(is.null(obj$DMats)) obj$DMats = getZDistances(obj)
  
  pb = txtProgressBar(0,sLen,style=3);count = 0
  for(it in thSeq){
    sr = matrix(obj$sender[,it],n,n) + matrix(obj$receiver[,it],n,n,byrow=TRUE)
    Mu = sr - obj$DMats[,,it]
    for(i in 1:n){
      tau = exp(-obj$beta1[it]*obj$DMats[i,,it])
      if(i %in% k.index){
        ind = which(k.index==i)
        AOut[i,] = AOut[i,] +
          pnorm(sqrt(tau*obj$beta0k[ind,it])*(Mu[i,]+obj$alphak[ind,it]))/sLen
        AIn[,i]  = AIn[,i] +
          pnorm(sqrt(tau*obj$beta0k[ind,it])*(Mu[,i]+obj$alphak[ind,it]))/sLen
      }else{
        # AOut[i,] = AOut[i,] +
        #   pnorm(sqrt(tau*mean(obj$beta0k[,it]))*(Mu[i,]+obj$alpha[it]))/sLen
        # AIn[,i]  = AIn[,i] +
        #   pnorm(sqrt(tau*mean(obj$beta0k[,it]))*(Mu[,i]+obj$alpha[it]))/sLen
        beta0k.i = rgamma(1,shape=obj$theta[it]/2,rate=obj$theta[it]/2)
        alphak.i = rnorm(1,obj$alpha[it],sqrt(obj$sigma2[it]))
        AOut[i,] = pnorm(sqrt(tau*beta0k.i)*(Mu[i,]+alphak.i))
        AIn[,i]  = pnorm(sqrt(tau*beta0k.i)*(Mu[,i]+alphak.i))
      }
    }
    count = count + 1
    setTxtProgressBar(pb,count)
  }
  
  
  Imin = matrix(1,n,n)-diag(n)
  return(list(rowDominated = AOut*Imin,
              colDominated = AIn*Imin,
              IntersectionRule = AOut*AIn*Imin,
              UnionRule = (AOut + AIn - AOut*AIn)*Imin))
}
