#' Deviance Information Criterion for the LSMNP
#'
#' This function computes the DIC for the LSMNP model. 
#'
#' Returns the DIC and effective number of parameter. Make sure to first run 'postProcessZ()' and attach to LSMNP obj 'ZProcessed' and 'BzProcessed'
#' 
#' @param A A list of nxn adjacency matrices
#' @param k.index A Kx1 integer vector giving the IDs of the respondents.  Should be the same length as A.
#' @param obj An object returned from LSMNPGibbs()
#' @param obj.pm (optional) An object returned from getPostMean()
#' @param burnin Integer. The number of initial MCMC iterations to throw out.
#' @param thin The number by which to thin the MCMC chain.
#' @return A list giving the DIC and pD, the effective number of parameters.
DIC.LSMNP = function(A,k.index = 1:length(A),obj,obj.pm=NULL,burnin=0,thin=1){
  
  thSeq = seq(burnin+1,length(obj$logPostTrace),by=thin)
  p = ifelse(length(dim(obj$Z))>2, dim(obj$Z)[2],1)
  Aind = list()
  for(k in 1:length(k.index)){
    Aind[[k]] = list()
    Aind[[k]][[1]] = which(A[[k]]==1,arr.ind=T)
    Aind[[k]][[2]] = which(A[[k]]==0,arr.ind=T)
    Aind[[k]][[2]] = Aind[[k]][[2]][-which(Aind[[k]][[2]][,1]==Aind[[k]][[2]][,2]),]
  }
  
  if(is.null(obj.pm)){
    obj.pm = getPostMean(obj,burnin,type="Dist",thin=thin)
  }
  if(is.null(obj.pm$DMat)) obj.pm = getPostMean(obj,burnin,type="Dist",thin=thin)
  
  if(is.null(obj$DMats)) obj$DMats = getZDistances(obj)
  
  logLikVec = numeric(length(thSeq))
  for(it in 1:length(thSeq)){
    logLikVec[it] = 
      LogLik(A,Aind,k.index,
             alphak=obj$alphak[,thSeq[it]],
             ss=obj$sender[,thSeq[it]],
             rr=obj$receiver[,thSeq[it]],
             Z=obj$DMats[,,thSeq[it]],
             b0k=obj$beta0k[,thSeq[it]],
             b1=obj$beta1[thSeq[it]],
             tau=NULL)
  }
  
  
  DBar = -2*mean(logLikVec)
  DThetaBar = -2*LogLik(A,Aind,k.index,obj.pm$alphak,obj.pm$sender,obj.pm$receiver,
                        obj.pm$DMat,obj.pm$beta0k,obj.pm$beta1)
  pD = DBar - DThetaBar
  
  return(list(DIC = pD + DBar, pD = pD))
}
